/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.resource.transport.aws.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Optional;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.gradle.api.credentials.AwsCredentials;
import org.gradle.internal.resource.PasswordCredentials;
import org.gradle.internal.resource.ResourceException;
import org.gradle.internal.resource.transport.aws.s3.S3ConnectionProperties;
import org.gradle.internal.resource.transport.aws.s3.S3RegionalResource;
import org.gradle.internal.resource.transport.http.HttpProxySettings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class S3Client {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3Client.class);
    private static final Pattern FILENAME_PATTERN = Pattern.compile("[^\\/]+\\.*$");
    private AmazonS3Client amazonS3Client;
    private final S3ConnectionProperties s3ConnectionProperties;

    public S3Client(AmazonS3Client amazonS3Client, S3ConnectionProperties s3ConnectionProperties) {
        this.s3ConnectionProperties = s3ConnectionProperties;
        this.amazonS3Client = amazonS3Client;
    }

    public S3Client(AwsCredentials awsCredentials, S3ConnectionProperties s3ConnectionProperties) {
        this.s3ConnectionProperties = s3ConnectionProperties;
        BasicAWSCredentials credentials = awsCredentials == null ? null : new BasicAWSCredentials(awsCredentials.getAccessKey(), awsCredentials.getSecretKey());
        this.amazonS3Client = this.createAmazonS3Client((AWSCredentials)credentials);
    }

    private AmazonS3Client createAmazonS3Client(AWSCredentials credentials) {
        AmazonS3Client amazonS3Client = new AmazonS3Client(credentials, this.createConnectionProperties());
        S3ClientOptions clientOptions = new S3ClientOptions();
        Optional<URI> endpoint = this.s3ConnectionProperties.getEndpoint();
        if (endpoint.isPresent()) {
            amazonS3Client.setEndpoint(((URI)endpoint.get()).toString());
            clientOptions.withPathStyleAccess(true);
        }
        amazonS3Client.setS3ClientOptions(clientOptions);
        return amazonS3Client;
    }

    private ClientConfiguration createConnectionProperties() {
        Optional<Integer> maxErrorRetryCount;
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        Optional<HttpProxySettings.HttpProxy> proxyOptional = this.s3ConnectionProperties.getProxy();
        if (proxyOptional.isPresent()) {
            HttpProxySettings.HttpProxy proxy = (HttpProxySettings.HttpProxy)this.s3ConnectionProperties.getProxy().get();
            clientConfiguration.setProxyHost(proxy.host);
            clientConfiguration.setProxyPort(proxy.port);
            PasswordCredentials credentials = proxy.credentials;
            if (credentials != null) {
                clientConfiguration.setProxyUsername(credentials.getUsername());
                clientConfiguration.setProxyPassword(credentials.getPassword());
            }
        }
        if ((maxErrorRetryCount = this.s3ConnectionProperties.getMaxErrorRetryCount()).isPresent()) {
            clientConfiguration.setMaxErrorRetry(((Integer)maxErrorRetryCount.get()).intValue());
        }
        return clientConfiguration;
    }

    public void put(InputStream inputStream, Long contentLength, URI destination) {
        try {
            S3RegionalResource s3RegionalResource = new S3RegionalResource(destination);
            String bucketName = s3RegionalResource.getBucketName();
            String s3BucketKey = s3RegionalResource.getKey();
            this.configureClient(s3RegionalResource);
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentLength(contentLength.longValue());
            PutObjectRequest putObjectRequest = new PutObjectRequest(bucketName, s3BucketKey, inputStream, objectMetadata);
            LOGGER.debug("Attempting to put resource:[{}] into s3 bucket [{}]", (Object)s3BucketKey, (Object)bucketName);
            this.amazonS3Client.putObject(putObjectRequest);
        }
        catch (AmazonClientException e) {
            throw ResourceException.putFailed((URI)destination, (Throwable)e);
        }
    }

    public S3Object getMetaData(URI uri) {
        LOGGER.debug("Attempting to get s3 meta-data: [{}]", (Object)uri.toString());
        return this.doGetS3Object(uri, true);
    }

    public S3Object getResource(URI uri) {
        LOGGER.debug("Attempting to get s3 resource: [{}]", (Object)uri.toString());
        return this.doGetS3Object(uri, false);
    }

    public List<String> list(URI parent) {
        ArrayList<String> results = new ArrayList<String>();
        S3RegionalResource s3RegionalResource = new S3RegionalResource(parent);
        String bucketName = s3RegionalResource.getBucketName();
        String s3BucketKey = s3RegionalResource.getKey();
        this.configureClient(s3RegionalResource);
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(bucketName).withPrefix(s3BucketKey).withMaxKeys(Integer.valueOf(1000)).withDelimiter("/");
        ObjectListing objectListing = this.amazonS3Client.listObjects(listObjectsRequest);
        results.addAll(this.resolveResourceNames(objectListing));
        while (objectListing.isTruncated()) {
            objectListing = this.amazonS3Client.listNextBatchOfObjects(objectListing);
            results.addAll(this.resolveResourceNames(objectListing));
        }
        return results;
    }

    private List<String> resolveResourceNames(ObjectListing objectListing) {
        ArrayList<String> results = new ArrayList<String>();
        List objectSummaries = objectListing.getObjectSummaries();
        if (null != objectSummaries) {
            for (S3ObjectSummary objectSummary : objectSummaries) {
                String key = objectSummary.getKey();
                String fileName = this.extractResourceName(key);
                if (null == fileName) continue;
                results.add(fileName);
            }
        }
        return results;
    }

    private String extractResourceName(String key) {
        Matcher matcher = FILENAME_PATTERN.matcher(key);
        if (matcher.find()) {
            String group = matcher.group(0);
            return group.contains(".") ? group : null;
        }
        return null;
    }

    private S3Object doGetS3Object(URI uri, boolean isLightWeight) {
        S3RegionalResource s3RegionalResource = new S3RegionalResource(uri);
        String bucketName = s3RegionalResource.getBucketName();
        String s3BucketKey = s3RegionalResource.getKey();
        this.configureClient(s3RegionalResource);
        GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, s3BucketKey);
        if (isLightWeight) {
            getObjectRequest.setRange(0L, 0L);
        }
        try {
            return this.amazonS3Client.getObject(getObjectRequest);
        }
        catch (AmazonServiceException e) {
            String errorCode = e.getErrorCode();
            if (null != errorCode && errorCode.equalsIgnoreCase("NoSuchKey")) {
                return null;
            }
            throw ResourceException.getFailed((URI)uri, (Throwable)e);
        }
    }

    private void configureClient(S3RegionalResource s3RegionalResource) {
        if (this.s3ConnectionProperties.getEndpoint().isPresent()) {
            this.amazonS3Client.setEndpoint(((URI)this.s3ConnectionProperties.getEndpoint().get()).toString());
        } else {
            this.amazonS3Client.setRegion(s3RegionalResource.getRegion());
        }
    }
}

