/*
 * Decompiled with CFR 0.152.
 */
package git4idea.update;

import com.intellij.dvcs.DvcsUtil;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vcs.VcsNotifier;
import com.intellij.openapi.vfs.VfsUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.ArrayUtil;
import git4idea.GitLocalBranch;
import git4idea.GitRemoteBranch;
import git4idea.GitUtil;
import git4idea.GitVcs;
import git4idea.branch.GitBranchUtil;
import git4idea.commands.Git;
import git4idea.commands.GitCommandResult;
import git4idea.commands.GitLineHandlerAdapter;
import git4idea.repo.GitBranchTrackInfo;
import git4idea.repo.GitRemote;
import git4idea.repo.GitRepository;
import git4idea.repo.GitRepositoryManager;
import git4idea.update.GitFetchResult;
import git4idea.util.GitUIUtil;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class GitFetcher {
    private static final Logger LOG = Logger.getInstance(GitFetcher.class);
    private final Project myProject;
    private final GitRepositoryManager myRepositoryManager;
    private final ProgressIndicator myProgressIndicator;
    private final boolean myFetchAll;
    private final GitVcs myVcs;
    private final Collection<Exception> myErrors = new ArrayList<Exception>();

    public GitFetcher(Project project, ProgressIndicator progressIndicator, boolean fetchAll) {
        this.myProject = project;
        this.myProgressIndicator = progressIndicator;
        this.myFetchAll = fetchAll;
        this.myRepositoryManager = GitUtil.getRepositoryManager(this.myProject);
        this.myVcs = GitVcs.getInstance(project);
    }

    public GitFetchResult fetch(GitRepository repository) {
        GitFetchResult fetchResult = GitFetchResult.success();
        if (!this.myFetchAll) {
            return this.fetchCurrentRemote(repository);
        }
        fetchResult = this.fetchAll(repository, fetchResult);
        VfsUtil.markDirtyAndRefresh((boolean)false, (boolean)true, (boolean)false, (VirtualFile[])new VirtualFile[]{repository.getGitDir()});
        return fetchResult;
    }

    public GitFetchResult fetch(VirtualFile root, String remoteName, String branch) {
        GitRepository repository = (GitRepository)this.myRepositoryManager.getRepositoryForRoot(root);
        if (repository == null) {
            return GitFetcher.logError("Repository can't be null for " + root, ((Object)((Object)this.myRepositoryManager)).toString());
        }
        GitRemote remote = GitUtil.findRemoteByName(repository, remoteName);
        if (remote == null) {
            return GitFetcher.logError("Couldn't find remote with the name " + remoteName, null);
        }
        String url = remote.getFirstUrl();
        if (url == null) {
            return GitFetcher.logError("URL is null for remote " + remote.getName(), null);
        }
        return this.fetchRemote(repository, remote, url, branch);
    }

    private static GitFetchResult logError(String message, String additionalInfo) {
        String addInfo = additionalInfo != null ? "\n" + additionalInfo : "";
        LOG.error(message + addInfo);
        return GitFetchResult.error(message);
    }

    private GitFetchResult fetchCurrentRemote(GitRepository repository) {
        FetchParams fetchParams = GitFetcher.getFetchParams(repository);
        if (fetchParams.isError()) {
            return fetchParams.getError();
        }
        GitRemote remote = fetchParams.getRemote();
        String url = fetchParams.getUrl();
        return this.fetchRemote(repository, remote, url, null);
    }

    private GitFetchResult fetchRemote(GitRepository repository, GitRemote remote, String url, String branch) {
        return GitFetcher.fetchNatively(repository, remote, url, branch);
    }

    private GitFetchResult fetchCurrentBranch(GitRepository repository) {
        FetchParams fetchParams = GitFetcher.getFetchParams(repository);
        if (fetchParams.isError()) {
            return fetchParams.getError();
        }
        GitRemote remote = fetchParams.getRemote();
        String remoteBranch = fetchParams.getRemoteBranch().getNameForRemoteOperations();
        String url = fetchParams.getUrl();
        return GitFetcher.fetchNatively(repository, remote, url, remoteBranch);
    }

    private static FetchParams getFetchParams(GitRepository repository) {
        GitLocalBranch currentBranch = repository.getCurrentBranch();
        if (currentBranch == null) {
            String message = "Current branch can't be null here. \nRepository: " + repository;
            LOG.error(message);
            return new FetchParams(GitFetchResult.error(new Exception(message)));
        }
        GitBranchTrackInfo trackInfo = GitBranchUtil.getTrackInfoForBranch(repository, currentBranch);
        if (trackInfo == null) {
            String message = "Tracked info is null for branch " + currentBranch + "\n Repository: " + repository;
            LOG.error(message);
            return new FetchParams(GitFetchResult.error(new Exception(message)));
        }
        GitRemote remote = trackInfo.getRemote();
        String url = remote.getFirstUrl();
        if (url == null) {
            String message = "URL is null for remote " + remote.getName();
            LOG.error(message);
            return new FetchParams(GitFetchResult.error(new Exception(message)));
        }
        return new FetchParams(remote, trackInfo.getRemoteBranch(), url);
    }

    private GitFetchResult fetchAll(GitRepository repository, GitFetchResult fetchResult) {
        for (GitRemote remote : repository.getRemotes()) {
            String url = remote.getFirstUrl();
            if (url == null) {
                LOG.error("URL is null for remote " + remote.getName());
                continue;
            }
            GitFetchResult res = GitFetcher.fetchNatively(repository, remote, url, null);
            res.addPruneInfo(fetchResult.getPrunedRefs());
            fetchResult = res;
            if (fetchResult.isSuccess()) continue;
            break;
        }
        return fetchResult;
    }

    private static GitFetchResult fetchNatively(GitRepository repository, GitRemote remote, String url, String branch) {
        String[] stringArray;
        Git git = (Git)ServiceManager.getService(Git.class);
        if (branch != null) {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = GitFetcher.getFetchSpecForBranch(branch, remote.getName());
        } else {
            stringArray = ArrayUtil.EMPTY_STRING_ARRAY;
        }
        String[] additionalParams = stringArray;
        GitFetchPruneDetector pruneDetector = new GitFetchPruneDetector();
        GitCommandResult result = git.fetch(repository, url, remote.getName(), Collections.singletonList(pruneDetector), additionalParams);
        GitFetchResult fetchResult = result.success() ? GitFetchResult.success() : (result.cancelled() ? GitFetchResult.cancel() : GitFetchResult.error(result.getErrorOutputAsJoinedString()));
        fetchResult.addPruneInfo(pruneDetector.getPrunedRefs());
        return fetchResult;
    }

    private static String getRidOfPrefixIfExists(String branch) {
        if (branch.startsWith("refs/heads/")) {
            return branch.substring("refs/heads/".length());
        }
        return branch;
    }

    public static String getFetchSpecForBranch(String branch, String remoteName) {
        branch = GitFetcher.getRidOfPrefixIfExists(branch);
        return "refs/heads/" + branch + ":" + "refs/remotes/" + remoteName + "/" + branch;
    }

    public Collection<Exception> getErrors() {
        return this.myErrors;
    }

    public static void displayFetchResult(Project project, GitFetchResult result, String errorNotificationTitle, Collection<? extends Exception> errors) {
        if (result.isSuccess()) {
            VcsNotifier.getInstance((Project)project).notifySuccess("Fetched successfully" + result.getAdditionalInfo());
        } else if (result.isCancelled()) {
            VcsNotifier.getInstance((Project)project).notifyMinorWarning("", "Fetch cancelled by user" + result.getAdditionalInfo());
        } else if (result.isNotAuthorized()) {
            String description;
            String title;
            if (errorNotificationTitle != null) {
                title = errorNotificationTitle;
                description = "Fetch failed: couldn't authorize";
            } else {
                title = "Fetch failed";
                description = "Couldn't authorize";
            }
            description = description + result.getAdditionalInfo();
            GitUIUtil.notifyMessage(project, title, description, true, null);
        } else {
            GitVcs instance = GitVcs.getInstance(project);
            if (instance != null && instance.getExecutableValidator().isExecutableValid()) {
                GitUIUtil.notifyMessage(project, "Fetch failed", result.getAdditionalInfo(), true, errors);
            }
        }
    }

    public boolean fetchRootsAndNotify(Collection<GitRepository> roots, String errorNotificationTitle, boolean notifySuccess) {
        String addInfo;
        HashMap<VirtualFile, String> additionalInfo = new HashMap<VirtualFile, String>();
        for (GitRepository repository : roots) {
            LOG.info("fetching " + repository);
            GitFetchResult result = this.fetch(repository);
            String ai = result.getAdditionalInfo();
            if (!StringUtil.isEmptyOrSpaces((String)ai)) {
                additionalInfo.put(repository.getRoot(), ai);
            }
            if (result.isSuccess()) continue;
            ArrayList<Exception> errors = new ArrayList<Exception>(this.getErrors());
            errors.addAll(result.getErrors());
            GitFetcher.displayFetchResult(this.myProject, result, errorNotificationTitle, errors);
            return false;
        }
        if (notifySuccess) {
            VcsNotifier.getInstance((Project)this.myProject).notifySuccess("Fetched successfully");
        }
        if (!StringUtil.isEmptyOrSpaces((String)(addInfo = this.makeAdditionalInfoByRoot(additionalInfo)))) {
            VcsNotifier.getInstance((Project)this.myProject).notifyMinorInfo("Fetch details", addInfo);
        }
        return true;
    }

    private String makeAdditionalInfoByRoot(Map<VirtualFile, String> additionalInfo) {
        if (additionalInfo.isEmpty()) {
            return "";
        }
        StringBuilder info = new StringBuilder();
        if (this.myRepositoryManager.moreThanOneRoot()) {
            for (Map.Entry<VirtualFile, String> entry : additionalInfo.entrySet()) {
                info.append(entry.getValue()).append(" in ").append(DvcsUtil.getShortRepositoryName((Project)this.myProject, (VirtualFile)entry.getKey())).append("<br/>");
            }
        } else {
            info.append(additionalInfo.values().iterator().next());
        }
        return info.toString();
    }

    private static class FetchParams {
        private GitRemote myRemote;
        private GitRemoteBranch myRemoteBranch;
        private GitFetchResult myError;
        private String myUrl;

        FetchParams(GitFetchResult error) {
            this.myError = error;
        }

        FetchParams(GitRemote remote, GitRemoteBranch remoteBranch, String url) {
            this.myRemote = remote;
            this.myRemoteBranch = remoteBranch;
            this.myUrl = url;
        }

        boolean isError() {
            return this.myError != null;
        }

        public GitFetchResult getError() {
            return this.myError;
        }

        public GitRemote getRemote() {
            return this.myRemote;
        }

        public GitRemoteBranch getRemoteBranch() {
            return this.myRemoteBranch;
        }

        public String getUrl() {
            return this.myUrl;
        }
    }

    private static class GitFetchPruneDetector
    extends GitLineHandlerAdapter {
        private static final Pattern PRUNE_PATTERN = Pattern.compile("\\s*x\\s*\\[deleted\\].*->\\s*(\\S*)");
        private final Collection<String> myPrunedRefs = new ArrayList<String>();

        private GitFetchPruneDetector() {
        }

        @Override
        public void onLineAvailable(String line, Key outputType) {
            Matcher matcher = PRUNE_PATTERN.matcher(line);
            if (matcher.matches()) {
                this.myPrunedRefs.add(matcher.group(1));
            }
        }

        public Collection<String> getPrunedRefs() {
            return this.myPrunedRefs;
        }
    }
}

